/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse AUTO_MOUSE

/////////////////////////////////////////////////////////////////////////////////

// Simple "Automatic Mouse". Simulates scanning the mouse over the full range of
// the screen with the X and Y scanning frequencies being different. TheEmu.

#define MOUSE_SPEED vec2(0.5,0.577777) * 0.2
#define MOUSE_POS   vec2((1.0+cos(iGlobalTime*MOUSE_SPEED))*u_WindowSize/2.0)
#define MOUSE_PRESS vec2(0.0,0.0)
#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )

/////////////////////////////////////////////////////////////////////////////////

#define MARCH_STEPS 128.0
#define MARCH_EPSILON 0.001
#define SHADOW_STEPS 40.0
#define SHADOW_EPSILON 0.01
#define GRADIENT_STEP 0.02

#define M_PI 3.1415926535897932384626433832795

float dot2( in vec3 v ) { return dot(v,v); }

float torus( vec3 p, vec2 t )
{
  vec2 q = vec2(length(p.xz)-t.x,p.y);
  return length(q)-t.y;
}

float sphere( vec3 p, float s )
{
  return length(p)-s;
}

float box(vec3 p, vec3 b, float roundness)
{
	return length(max(abs(p)-(b-vec3(roundness)),0.0))-roundness;
}

float cylinder( vec3 p, vec3 c )
{
  return length(p.xz-c.xy)-c.z;
}

float cone( vec3 p, vec2 c )
{
    float q = length(p.xy);
    return dot(c,vec2(q,p.z));
}

float plane( vec3 p, vec4 n )
{
  return dot(p,n.xyz) + n.w;
}

float hexPrism( vec3 p, vec2 h )
{
    vec3 q = abs(p);
    return max(q.z-h.y,max((q.x*0.866025+q.y*0.5),q.y)-h.x);
}

float triPrism( vec3 p, vec2 h )
{
    vec3 q = abs(p);
    return max(q.z-h.y,max(q.x*0.866025+p.y*0.5,-p.y)-h.x*0.5);
}

float capsule( vec3 p, vec3 a, vec3 b, float r )
{
    vec3 pa = p - a, ba = b - a;
    float h = clamp( dot(pa,ba)/dot(ba,ba), 0.0, 1.0 );
    return length( pa - ba*h ) - r;
}

float cappedCylinder( vec3 p, vec2 h )
{
  vec2 d = abs(vec2(length(p.xz),p.y)) - h;
  return min(max(d.x,d.y),0.0) + length(max(d,0.0));
}

float cappedCone( in vec3 p, in vec3 c )
{
    vec2 q = vec2( length(p.xz), p.y );
    vec2 v = vec2( c.z*c.y/c.x, -c.z );
    vec2 w = v - q;
    vec2 vv = vec2( dot(v,v), v.x*v.x );
    vec2 qv = vec2( dot(v,w), v.x*w.x );
    vec2 d = max(qv,0.0)*qv/vv;
    return sqrt( dot(w,w) - max(d.x,d.y) ) * sign(max(q.y*v.x-q.x*v.y,w.y));
}

float ellipsoid( in vec3 p, in vec3 r )
{
    return (length( p/r ) - 1.0) * min(min(r.x,r.y),r.z);
}

float triangle( vec3 p, vec3 a, vec3 b, vec3 c )
{
    vec3 ba = b - a; vec3 pa = p - a;
    vec3 cb = c - b; vec3 pb = p - b;
    vec3 ac = a - c; vec3 pc = p - c;
    vec3 nor = cross( ba, ac );

    return sqrt(
    (sign(dot(cross(ba,nor),pa)) +
     sign(dot(cross(cb,nor),pb)) +
     sign(dot(cross(ac,nor),pc))<2.0)
     ?
     min( min(
     dot2(ba*clamp(dot(ba,pa)/dot2(ba),0.0,1.0)-pa),
     dot2(cb*clamp(dot(cb,pb)/dot2(cb),0.0,1.0)-pb) ),
     dot2(ac*clamp(dot(ac,pc)/dot2(ac),0.0,1.0)-pc) )
     :
     dot(nor,pa)*dot(nor,pa)/dot2(nor) );
}

float quad( vec3 p, vec3 a, vec3 b, vec3 c, vec3 d )
{
    vec3 ba = b - a; vec3 pa = p - a;
    vec3 cb = c - b; vec3 pb = p - b;
    vec3 dc = d - c; vec3 pc = p - c;
    vec3 ad = a - d; vec3 pd = p - d;
    vec3 nor = cross( ba, ad );

    return sqrt(
    (sign(dot(cross(ba,nor),pa)) +
     sign(dot(cross(cb,nor),pb)) +
     sign(dot(cross(dc,nor),pc)) +
     sign(dot(cross(ad,nor),pd))<3.0)
     ?
     min( min( min(
     dot2(ba*clamp(dot(ba,pa)/dot2(ba),0.0,1.0)-pa),
     dot2(cb*clamp(dot(cb,pb)/dot2(cb),0.0,1.0)-pb) ),
     dot2(dc*clamp(dot(dc,pc)/dot2(dc),0.0,1.0)-pc) ),
     dot2(ad*clamp(dot(ad,pd)/dot2(ad),0.0,1.0)-pd) )
     :
     dot(nor,pa)*dot(nor,pa)/dot2(nor) );
}

// --- //

float smin( float a, float b, float k )
{
    float h = clamp(0.5+0.5*(b-a)/k,0.0,1.0);
    return mix(b,a,h)-k*h*(1.0-h);
}

// ----------------------------------------------- //
// --- NOTES ON DISTANCE FIELD TRANSFORMATIONS --- //
// ----------------------------------------------- //

// UNION:		min(d1,d2)
// SUBTRACT:	max(-d1,d2)
// INTERSECT:	max(d1,d2)
// REPEAT:		map(mod(p,c)-0.5*c)
// SCALE:		map(p/s)*s
// ROTATE:		map(invert(m)*p)	// where m == 4x4 rot matrix
/*

float twist( vec3 p )
{
    float c = cos(20.0*p.y);
    float s = sin(20.0*p.y);
    mat2  m = mat2(c,-s,s,c);
    vec3  q = vec3(m*p.xz,p.y);
    return primitive(q);
}

float bend( vec3 p )
{
    float c = cos(20.0*p.y);
    float s = sin(20.0*p.y);
    mat2  m = mat2(c,-s,s,c);
    vec3  q = vec3(m*p.xy,p.z);
    return primitive(q);
}

*/

// ----------------------------------------------- //
// ----------------------------------------------- //
// ----------------------------------------------- //

// field of view of camera
#define FOV 0.3
float CAMERA_DISTANCE = 30.*(1.+sin(iGlobalTime*2.*M_PI*0.05));
// generate combined distance field
float map(vec3 p)
{
	// draw rounded boxes
    float qs = (cos(iGlobalTime/11.)+2.)/3.;
    vec3 q = p;
    q.xyz = mod(p.xyz,vec3(10.0*qs))-5.*qs;
    float boxSize = 4.*qs;
	float d = box(q,vec3(boxSize), boxSize/2.);

    // sub mini spheres from rounded boxes
    float qr = (sin(iGlobalTime/1.)+4.)/5.;
    vec3 r = p;
    r.xyz = mod(r.xyz,vec3(2.*qr))-1.*qr;
    d = max(-sphere(r,1.2*qr), d);
    
    // intersect previous geometry with megasphere
    d = max(d,sphere(p,30.));
    
    // subtract smaller sphere from megasphere
    d = max(d,-sphere(p,25.));

	return d;
}

// raymarching procedure with lots of output info
void march(vec3 origin, vec3 direction, out vec3 rayHead, out bool hit, out float dist, out float steps)
{
    float distanceTraveled = 0.;
    hit = true;
    for(float step=0.; step<MARCH_STEPS; step++)
    {
        rayHead = origin + (direction*distanceTraveled);
        dist = map(rayHead);
        steps = step;
        if(dist<MARCH_EPSILON||step>=MARCH_STEPS) return;
        distanceTraveled += dist;
    }
    hit = false;
}

// get origin and direction of ray to be cast for camera/screen xy (pixel) coordinate
void cameraRay(vec2 scanLines, vec3 cameraPosition, vec3 target, float fov, out vec3 pos, out vec3 dir)
{
	vec3 forward = normalize(target-cameraPosition);
    vec3 up = vec3(0.,1.,0.);
    vec3 right = normalize(cross(forward, up));
    up = normalize(cross(forward,right));
    
    right = right*scanLines.x*fov;
    up = up*scanLines.y*fov;
    
    pos = cameraPosition;
    dir = (right+up+forward);
}

// get surface normal of point on distance field for shading
vec3 gradient( vec3 pos )
{
	const vec3 dx = vec3( GRADIENT_STEP, 0.0, 0.0 );
	const vec3 dy = vec3( 0.0, GRADIENT_STEP, 0.0 );
	const vec3 dz = vec3( 0.0, 0.0, GRADIENT_STEP );
	return normalize(vec3(
		map( pos + dx ) - map( pos - dx ),
		map( pos + dy ) - map( pos - dy ),
		map( pos + dz ) - map( pos - dz )
	));
}

// calculate diffuse lighting for point on distance field given surface normal
float diffuse(vec3 normal, vec3 lightPosition)
{
	return max(dot(normal,normalize(lightPosition)),0.);
}

// calculate diffuse lighting for point on distance field
float softshadow( in vec3 ro, in vec3 rd, in float mint, in float tmax )
{
	float res = 1.0;
    float t = mint;
    for( int i=0; i<int(SHADOW_STEPS); i++ )
    {
		float h = map( ro + rd*t );
        res = min( res, 8.0*h/t );
        t += clamp( h, 0.02, 0.10 );
        if( h<SHADOW_EPSILON || t>tmax ) break;
    }
    return clamp( res, 0.0, 1.0 );
}

// calculate combined lighting for point on distance field
float shadePoint(vec3 position, vec3 normal, vec3 lightPosition)
{
    float diffuseLight = diffuse(normal, lightPosition);
	// speculative light - it might be specular, but we're not really sure
    float specularLight = pow(diffuseLight,50.);
	float shadow = softshadow(position, normalize(lightPosition), 0.02, 2.25);
    return (diffuseLight*0.7+specularLight*3.0)*shadow;
}

// generate background color
vec3 background(vec2 uv)
{
	return vec3(0.5,0.5,0.5)*uv.y;
}

// compose image
void mainImage(out vec4 fragColor, in vec2 fragCoord )
{
    // scale coordinates to -1 -> +1, adjusted for screen aspect ratio
	vec2 uv = fragCoord.xy / iResolution.xy;
	uv = uv*2.-1.;
    uv.x = uv.x*iResolution.x/iResolution.y;
    
    // get mouse coords
    vec2 mouse = iMouse.xy/iResolution.xy;
    
    // get ray representing camera scanline position and direction
    float cameraRotationSpeed = 0.2;
    vec3 cameraPosition = normalize(vec3(sin(iGlobalTime*cameraRotationSpeed),
                                         sin(iGlobalTime*cameraRotationSpeed),
                                         cos(iGlobalTime*cameraRotationSpeed)));
    if(iMouse.z>0.)
        cameraPosition = normalize(vec3(	sin(mouse.x*2.*M_PI),
                                        	(mouse.y*2.-1.),
                                        	cos(mouse.x*2.*M_PI)));
    cameraPosition *= CAMERA_DISTANCE;
    vec3 cameraOrigin;
    vec3 cameraDirection;
    cameraRay(uv, cameraPosition, vec3(0.,0.,0.), FOV, cameraOrigin, cameraDirection);
    
    // calculate distance, hit point, etc of distance field via raymarching
    vec3 rayHead;
    bool hit;
    float dist;
    float steps = 40.;
    march(cameraOrigin, cameraDirection, rayHead, hit, dist, steps);
    
    // calculate normal for point on distance field
    vec3 normal = gradient(rayHead);
    
    // calculate light for point
	vec3 rotLightPos = vec3(-1.);
    vec3 latRotLightPos = vec3(1.);
    vec3 lightPos = vec3(cos(iGlobalTime),sin(iGlobalTime),cos(iGlobalTime));
    float light = 0.;
    light += shadePoint(rayHead, normal, rotLightPos);
    light += shadePoint(rayHead, normal, latRotLightPos);
    light += shadePoint(rayHead, normal, lightPos);
    light /= 2.;
    
    // specify color of entire distance field
    vec3 fieldColor = vec3(1.,0.8,0.5);
    
    // final steps and background
    vec3 color = fieldColor*light;
    if(!hit) color = background(uv);
    
    // set fragment shader color
    fragColor = vec4(color,1.0);

}

void main ( void )
{
   mainImage ( gl_FragColor, gl_FragCoord.xy );
}

